from enum import IntEnum


class TriggerSignal(IntEnum):
    TRF = 256
    SamplePD4 = 128
    SamplePD3 = 64
    SamplePD2 = 32
    SamplePD1 = 16
    SamplePMT3 = 4
    SamplePMT2 = 2
    SamplePMT1 = 1


class OutputSignal(IntEnum):
    Flash = 1 << 23
    Alpha = 1 << 22
    HTS_Alpha = 1 << 21
    InputGatePMT2 = 1 << 17
    InputGatePMT1 = 1 << 16
    HVGatePMT2 = 1 << 13
    HVGatePMT1 = 1 << 12
    HVOnPMT3 = 1 << 10
    HVOnPMT2 = 1 << 9
    HVOnPMT1 = 1 << 8
    Deprc_IntRstPD4 = 1 << 7
    Deprc_IntRstPD3 = 1 << 6
    Deprc_IntRstPD2 = 1 << 5
    Deprc_IntRstPD1 = 1 << 4
    Deprc_IntRstPMT2 = 1 << 1
    Deprc_IntRstPMT1 = 1


class MeasurementChannel(IntEnum):
    PMT1 = 0
    PMT2 = 1
    PMT3 = 2
    PMT4 = 3
    PD1 = 4
    PD2 = 5
    PD3 = 6
    PD4 = 7


class IntegratorMode(IntEnum):
    full_reset = 0b010
    low_range_reset = 0b011
    integrate_autorange = 0b100
    integrate_with_fixed_range = 0b101
    integrate_in_low_range = 0b110
    integrate_in_high_range = 0b111


class AnalogControlMode(IntEnum):
    add_to_high_offset = 0b001
    add_to_low_offset = 0b010
    add_to_low_and_high_offset = 0b011
    full_offset_reset = 0b100
    high_offset_reset = 0b101
    low_offset_reset = 0b110
    read_offset = 0b111


class meas_seq_generator:

    def __init__(self):
        self.currSequence = []

    def GetSequenceByteArray(self):
        retArray = bytearray()
        for value in self.currSequence:
            bytesAdded = int(value, 0).to_bytes(4, 'big')
            for aByte in bytesAdded:
                retArray.append(aByte)

        return retArray

    def Stop(self, returnValue):
        cmd = returnValue & 0x0F
        self.currSequence.append(cmd)

    def SubSequence(self, relative, address):
        cmd = (0x06 << 24) + (address & 0x0FFF)
        if relative:
            cmd = cmd | 0x400000
        self.currSequence.append(cmd)

    def Return(self):
        cmd = 0x04 << 24
        self.currSequence.append(cmd)

    def Loop(self, iterations):
        if (iterations - 1) > 0xFFFF:
            raise Exception("FPGA Parameter error. Too long loop")
        cmd = (0x07 << 24) + ((iterations - 1) & 0xFFFF)
        self.currSequence.append(cmd)

    def LoopEnd(self):
        cmd = 0x05 << 24
        self.currSequence.append(cmd)

    def TimerWaitAndRestart(self, delay):
        if delay > 0:
            delay = delay - 1
        if delay > 0x3FFFFFF:
            raise Exception("FPGA Parameter error. Too long TimerWait")
        cmd = (0x7C << 24) + (delay & 0x3FFFFFF)
        self.currSequence.append(cmd)

    def TimerWait(self):
        self.TimerWaitAndRestart(0)

    def Delay(self, delay):
        if delay > 0:
            delay = delay - 1
        if delay > 0xFFFF:
            raise Exception("FPGA Parameter error. Too long delay")
        cmd = (0x78 << 24) + (delay & 0xFFFF)
        self.currSequence.append(cmd)

    def StepCounterWaitAndRestart(self, quarterMicrosteps):
        cmd = 0x6C000000 | int(quarterMicrosteps)
        self.currSequence.append(cmd)

    def WaitForTriggerInput(self, TRF_lo=False, TRF_hi=False):
        cmd = (0x74 << 24)
        if TRF_hi:
            cmd = cmd + 1
        if TRF_lo:
            cmd = cmd + 2
        self.currSequence.append(cmd)

    def SetTriggerOutput(self, triggerSignals):
        cmd = (0x01 << 24) + triggerSignals
        self.currSequence.append(cmd)

    def SetIntegratorMode(self, pd4=0, pd3=0, pd2=0, pd1=0, pmt2=0, pmt1=0):
        cmd = (0b00001111 << 24)
        cmd = cmd + (pd4 << 21)
        cmd = cmd + (pd3 << 18)
        cmd = cmd + (pd2 << 15)
        cmd = cmd + (pd1 << 12)
        cmd = cmd + (pmt2 << 3)
        cmd = cmd + pmt1
        self.currSequence.append(cmd)

    def SetAnalogControl(self, pd4=0, pd3=0, pd2=0, pd1=0, pmt2=0, pmt1=0):
        cmd = (0b00001110 << 24)
        cmd = cmd + (pd4 << 21)
        cmd = cmd + (pd3 << 18)
        cmd = cmd + (pd2 << 15)
        cmd = cmd + (pd1 << 12)
        cmd = cmd + (pmt2 << 3)
        cmd = cmd + pmt1
        self.currSequence.append(cmd)

    def SetSignals(self, outputSignals):
        cmd = (0x02 << 24) + outputSignals
        self.currSequence.append(cmd)

    def ResetSignals(self, outputSignals):
        cmd = (0x03 << 24) + outputSignals
        self.currSequence.append(cmd)

    def GetAnalogResult(self, channel, isRelativeAddr, ignoreRange, isHiRange, addResult, dword, addrPos, resultPos):
        cmd = 1 << 31
        cmd = cmd + ((channel & 0x07) << 24)
        if isRelativeAddr:
            cmd = cmd | (0x01 << 22)
        if ignoreRange:
            cmd = cmd | (0x01 << 21)
        if isHiRange:
            cmd = cmd | (0x01 << 20)
        if addResult:
            cmd = cmd | (0x01 << 19)
        if dword:
            cmd = cmd | (0x01 << 18)
        cmd = cmd + ((addrPos & 0x0F) << 12)
        cmd = cmd + (resultPos & 0x0FFF)
        self.currSequence.append(cmd)

    def PulseCounterControl(self, channel, cumulative, resetCounter, resetPresetCounter, correctionOn):
        cmd = 0x22 << 26
        cmd = cmd + ((channel & 0x03) << 24)
        cmd = cmd | (0x01 << 23)
        if cumulative:
            cmd = cmd | (0x01 << 22)
        if resetCounter:
            cmd = cmd | (0x01 << 21)
        if resetPresetCounter:
            cmd = cmd | (0x01 << 20)
        if correctionOn:
            cmd = cmd | (0x01 << 19)
        self.currSequence.append(cmd)

    def GetPulseCounterResult(self, channel, relative, resetCounter, cumulative, dword, addrPos, resultPos, deadTime=False):
        cmd = 1 << 31
        cmd = cmd | ((channel & 0x03) << 24)
        cmd = cmd | (0x01 << 23)
        if relative:
            cmd = cmd | (0x01 << 22)
        if deadTime:
            cmd = cmd | (0x01 << 21)
        if resetCounter:
            cmd = cmd | (0x01 << 20)
        if cumulative:
            cmd = cmd | (0x01 << 19)
        if dword:
            cmd = cmd | (0x01 << 18)
        cmd = cmd | ((addrPos & 0xF) << 12)
        cmd = cmd | (resultPos & 0x0FFF)
        self.currSequence.append(cmd)

    def SetAddrRegConditionally(self, channel, checkCounterValue, relative, overridePrevValue, checkHiRange, dataPosition, addressPosition, addrValue):
        cmd = 0x18 << 27
        cmd = cmd | ((channel & 0x07) << 24)
        if checkCounterValue:
            cmd = cmd | (0x01 << 23)
        if relative:
            cmd = cmd | (0x01 << 22)
        if overridePrevValue:
            cmd = cmd | (0x01 << 21)
        if checkHiRange:
            cmd = cmd | (0x01 << 20)
        cmd = cmd | ((dataPosition & 0xF) << 16)
        cmd = cmd | ((addressPosition & 0xF) << 12)
        cmd = cmd | (addrValue & 0x0FFF)
        self.currSequence.append(cmd)

    def SetAddrReg(self, relative, dataNotAddrSrc, sign, stackNotRegSrc, srcReg, dstReg, addr):
        cmd = 0x08 << 24
        if dataNotAddrSrc:
            cmd = cmd | (0x01 << 23)
        if relative:
            cmd = cmd | (0x01 << 22)
        if sign:
            cmd = cmd | (0x01 << 21)
        if stackNotRegSrc:
            cmd = cmd | (0x01 << 20)
        cmd = cmd | ((srcReg & 0x0F) << 16)
        cmd = cmd | ((dstReg & 0x0F) << 12)
        cmd = cmd | (addr & 0x0FFF)
        self.currSequence.append(cmd)

    def SetDataReg(self, stackNotRegSrc, add, sign, stackNotRegDst, srcReg, dstReg, value):
        cmd = 0x09 << 24
        if stackNotRegSrc:
            cmd = cmd | (0x01 << 23)
        if add:
            cmd = cmd | (0x01 << 22)
        if sign:
            cmd = cmd | (0x01 << 21)
        if stackNotRegDst:
            cmd = cmd | (0x01 << 20)
        cmd = cmd | ((srcReg & 0x0F) << 16)
        cmd = cmd | ((dstReg & 0x0F) << 12)
        cmd = cmd | (value & 0xFFF)
        self.currSequence.append(cmd)

    def LoadDataReg(self, stackNotRegDst, dstReg, value):
        cmd = 0b00001010 << 24
        cmd = cmd | (stackNotRegDst << 20)
        cmd = cmd | ((0xF & dstReg) << 16)
        cmd = cmd + (0xFFFF & value)
        self.currSequence.append(cmd)

    def ClearResultBuffer(self, relative, dword, addrReg, addr):
        cmd = 0x8C << 24
        if relative:
            cmd = cmd | (0x01 << 22)
        if dword:
            cmd = cmd | (0x01 << 18)
        cmd = cmd | ((addrReg & 0x0F) << 12)
        cmd = cmd | (addr & 0xFFF)
        self.currSequence.append(cmd)

    def ClearResultBufferRange(self, address=0, size=1):
        if (address < 0) or (size < 1) or ((address + size) > 4096):
            raise Exception(f"FPGA Parameter Error: Invalid address range.")
        self.SetAddrReg(relative=False, dataNotAddrSrc=False, sign=0, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=address)
        self.Loop(size)
        self.ClearResultBuffer(relative=True, dword=False, addrReg=0, addr=0)
        self.SetAddrReg(relative=True, dataNotAddrSrc=False, sign=0, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=1)
        self.LoopEnd()

